<?php
/**
 * Core of Marzban Sales Bot
 */

error_reporting(0);
ini_set('display_errors', 0);
date_default_timezone_set('Asia/Tehran');

// --- PATHS & JSON FILES ---
$configFile    = __DIR__ . '/config.json';
$prodFile      = __DIR__ . '/products.json';
$usersFile     = __DIR__ . '/users_data.json';
$settingFile   = __DIR__ . '/settings.json';
$stepFile      = __DIR__ . '/steps.json';
$panelsFile    = __DIR__ . '/panels.json';

if (!file_exists($configFile)) {
    exit("File config.json not found.");
}

// --- LOAD DATA (GLOBAL STATE) ---
$config   = json_decode(file_get_contents($configFile), true);
$products = file_exists($prodFile)   ? json_decode(file_get_contents($prodFile), true)   : [];
$db       = file_exists($usersFile)  ? json_decode(file_get_contents($usersFile), true)  : [
    "users" => [],
    "stats" => [
        "total_sales" => 0,
        "order_count" => 0
    ]
];
$panels   = file_exists($panelsFile) ? json_decode(file_get_contents($panelsFile), true) : [];
$settings = file_exists($settingFile)? json_decode(file_get_contents($settingFile), true): [
    "tutorial"        => "آموزشی ثبت نشده است.",
    "card_number"     => "ثبت نشده",
    "trial_mb"        => 100,
    "trial_hours"     => 1,
    "active_panel_idx"=> null,
    "aff_percent"     => 10
];
$steps    = file_exists($stepFile)   ? json_decode(file_get_contents($stepFile), true)   : [];

define('BOT_TOKEN', $config['bot_token']);
define('ADMIN_ID',  $config['admin_id']);

// --- SAVE HELPERS ---
function saveDB() {
    global $db, $usersFile;
    file_put_contents(
        $usersFile,
        json_encode($db, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)
    );
}

function saveSettings() {
    global $settings, $settingFile;
    file_put_contents(
        $settingFile,
        json_encode($settings, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)
    );
}

function saveProducts() {
    global $products, $prodFile;
    file_put_contents(
        $prodFile,
        json_encode($products, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)
    );
}

function saveSteps($s) {
    global $stepFile;
    file_put_contents($stepFile, json_encode($s));
}

function savePanels() {
    global $panels, $panelsFile;
    file_put_contents(
        $panelsFile,
        json_encode($panels, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)
    );
}

// --- TELEGRAM CORE ---
function bot($method, $datas = []) {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/" . $method;
    $ch  = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $datas);
    $res = curl_exec($ch);
    curl_close($ch);
    return json_decode($res, true);
}

// --- USER INIT / ENSURE ---
function ensureUser($uid) {
    global $db;
    if (!isset($db['users'][$uid])) {
        $db['users'][$uid] = [
            'balance'   => 0,
            'services'  => [],
            'trial'     => false,
            'inviter'   => null,
            'sub_count' => 0
        ];
        saveDB();
    }
}

// --- MENUS ---
function getMainMenu($uid) {
    $keyboard = [
        ["🛒 خرید اشتراک", "🎁 تست رایگان"],
        ["💰 کیف پول", "👤 سرویس‌های من"],
        ["👥 زیرمجموعه‌گیری", "📚 آموزش"],
        ["☎️ پشتیبانی"]
    ];
    if ($uid == ADMIN_ID) {
        $keyboard[] = ["⚙️ پنل مدیریت"];
    }
    return json_encode([
        'keyboard'        => $keyboard,
        'resize_keyboard' => true
    ]);
}

function getAdminMenu() {
    return json_encode([
        'keyboard' => [
            ["➕ اضافه کردن پنل", "🖥 مدیریت پنل‌ها"],
            ["➕ افزودن محصول", "❌ حذف محصول"],
            ["📊 آمار فروش", "📢 پیام همگانی"],
            ["💳 تنظیم کارت", "📚 تنظیم آموزش"],
            ["🎁 مدیریت تست", "🔙 بازگشت به منو"]
        ],
        'resize_keyboard' => true
    ]);
}

// --- MARZBAN API WRAPPER ---
class Marzban {
    public static function getToken($panel) {
        if (!$panel) return null;

        $url = rtrim($panel['url'], '/') . '/api/admin/token';
        $ch  = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
            'username' => $panel['user'],
            'password' => $panel['pass']
        ]));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);

        $res  = curl_exec($ch);
        $body = json_decode($res, true);
        return $body['access_token'] ?? null;
    }

    public static function call($panel, $endpoint, $method = "GET", $data = null, $token = null) {
        $url = rtrim($panel['url'], '/') . $endpoint;
        $ch  = curl_init($url);

        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        if ($token) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                "Authorization: Bearer $token",
                "Content-Type: application/json"
            ]);
        }
        if ($data) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }

        $res = curl_exec($ch);
        return json_decode($res, true);
    }
}

/* ------------ Helpers ------------- */

function getQrCodeUrl($text, $size = 300) {
    $encoded = urlencode($text);
    $sz      = intval($size);
    if ($sz < 100) $sz = 100;
    if ($sz > 800) $sz = 800;
    return "https://chart.googleapis.com/chart?cht=qr&chs={$sz}x{$sz}&chl={$encoded}";
}

function normalizeSubscriptionUrl($url) {
    return trim($url);
}

function notifyAdmin($text) {
    bot('sendMessage', [
        'chat_id' => ADMIN_ID,
        'text'    => $text
    ]);
}

function getPanelById($id) {
    global $panels;
    return isset($panels[$id]) ? $panels[$id] : null;
}

function deleteMarzbanUser($panelId, $username) {
    $panel = getPanelById($panelId);
    if (!$panel) return false;

    $token = Marzban::getToken($panel);
    if (!$token) return false;

    Marzban::call($panel, "/api/user/$username", "DELETE", null, $token);
    return true;
}

function getMarzbanUser($panelId, $username) {
    $panel = getPanelById($panelId);
    if (!$panel) return null;

    $token = Marzban::getToken($panel);
    if (!$token) return null;

    $res = Marzban::call($panel, "/api/user/$username", "GET", null, $token);
    return $res ?: null;
}

function formatBytes($bytes) {
    if ($bytes === null) return "نامشخص";
    $bytes = (float)$bytes;
    $units = ['B','KB','MB','GB','TB'];
    $i = 0;
    while ($bytes >= 1024 && $i < count($units)-1) {
        $bytes /= 1024;
        $i++;
    }
    return round($bytes, 2) . " " . $units[$i];
}

function formatRemainingTime($timestamp) {
    if (!$timestamp) return "نامشخص";
    $now = time();
    $diff = $timestamp - $now;
    if ($diff <= 0) return "منقضی شده";

    $days  = floor($diff / 86400);
    $hours = floor(($diff % 86400) / 3600);
    $mins  = floor(($diff % 3600) / 60);

    $parts = [];
    if ($days > 0)  $parts[] = $days . " روز";
    if ($hours > 0) $parts[] = $hours . " ساعت";
    if ($mins > 0)  $parts[] = $mins . " دقیقه";

    return implode(' و ', $parts);
}